import os
import re
from flask import Flask, request, render_template, redirect, session, url_for, flash, jsonify
from models import init_db, User, Post
from sqlalchemy.exc import IntegrityError
from functools import wraps

from bot import bot_fetch, bot_report

# Flags from environment
FLAG1 = os.environ.get("FLAG1", "FLAG{lv1_placeholder}")
FLAG2 = os.environ.get("FLAG2", "FLAG{lv2_placeholder}")
FLAG3 = os.environ.get("FLAG3", "FLAG{lv3_placeholder}")
FLAG4 = os.environ.get("FLAG4", "FLAG{lv4_placeholder}") # Level 3 flag, dont worry about it
app = Flask(__name__, template_folder="templates", static_folder="static")
app.config["SECRET_KEY"] = os.environ.get("SECRET_KEY", "gotham")

db = init_db()

def login_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        # Check if authenticated via flag
        auth_flag = request.cookies.get('FLAG')

        if auth_flag in [FLAG1, FLAG2, FLAG3, FLAG4]:
            # Authenticate as the corresponding villain
            # Lets not think about security here lol
            if auth_flag == FLAG1:
                session["user_id"] = get_or_create_villain("Bane")
            elif auth_flag == FLAG2:
                session["user_id"] = get_or_create_villain("Riddler")
            elif auth_flag == FLAG3:
                session["user_id"] = get_or_create_villain("Joker")
            elif auth_flag == FLAG4:
                session["user_id"] = get_or_create_villain("Mr. Freeze")
            return f(*args, **kwargs)
        
        # Check normal session authentication
        if "user_id" not in session:
            flash("You must log in to access that page.", "warning")
            return redirect(url_for("login"))
        return f(*args, **kwargs)
    return decorated_function

def get_or_create_villain(username):
    """Get or create villain user and return their ID"""
    villain = db.query(User).filter_by(username=username).first()
    if not villain:
        villain = User(username=username, password="admin_password")
        db.add(villain)
        db.commit()
    return villain.id

@app.route("/")
def index():
    return render_template("index.html")

@app.route("/register", methods=["GET", "POST"])
def register():
    if request.method == "POST":
        username = request.form["username"]
        password = request.form["password"]
        new_user = User(username=username, password=password)
        try:
            db.add(new_user)
            db.commit()
            flash("Registration successful! Please log in.", "success")
            return redirect(url_for("login"))
        except IntegrityError:
            db.rollback()
            flash("Username already taken. Please choose another.", "danger")
    return render_template("register.html")

@app.route("/login", methods=["GET", "POST"])
def login():
    if request.method == "POST":
        username = request.form["username"]
        password = request.form["password"]
        user = db.query(User).filter_by(username=username, password=password).first()
        if user:
            session["user_id"] = user.id
            flash("Login successful!", "success")
            return redirect(url_for("dashboard"))
        else:
            flash("Invalid credentials. Please try again.", "danger")
    return render_template("login.html")

@app.route("/dashboard", methods=["GET", "POST"])
@login_required
def dashboard():
    user = db.query(User).get(session["user_id"])
    posts = db.query(Post).filter(
        (Post.author_id == user.id) | (Post.recipient == user.username)
    ).all()

    flag_message = None
    flag_status = None

    # Handle new post submission
    if request.method == "POST" and "content" in request.form:
        recipient = request.form["recipient"]
        content = request.form["content"]
        new_post = Post(author_id=user.id, recipient=recipient, content=content)
        db.add(new_post)
        db.commit()
        flash("Post created successfully!", "success")
        return redirect(url_for("dashboard"))

    # Handle flag validation submission
    if request.method == "POST" and "flag" in request.form:
        submitted_flag = request.form["flag"].strip()
        if submitted_flag in [FLAG1, FLAG2, FLAG3, FLAG4]:
            flag_message = f"Valid flag! {submitted_flag} has been recognized."
            flag_status = "success"
        else:
            flag_message = f"Invalid flag. Gotham intelligence does not recognize it."
            flag_status = "danger"

    return render_template(
        "dashboard.html",
        user=user,
        posts=posts,
        FLAG1=FLAG1,
        FLAG2=FLAG2,
        FLAG3=FLAG3,
        FLAG4=FLAG4,
        flag_message=flag_message,
        flag_status=flag_status
    )


@app.route("/logout")
@login_required
def logout():
    session.pop("user_id", None)
    flash("You have been logged out.", "info")
    return redirect(url_for("index"))

@app.route("/create_post", methods=["GET", "POST"])
@login_required
def create_post():
    user = db.query(User).get(session["user_id"])

    villains = {
        "Bane": "Strong and silent type. He prefers direct communication, no frills.",
        "Riddler": "Master of puzzles and mind games, he loves to apply his arbitrary filters to all posts.",
        "Mr. Freeze": "A cold and calculating villain. He likes to freeze posts in time, preventing any further interaction.",
        "Joker": "A chaotic wildcard, unpredictable in his methods. He refuses to look at any any posts, even when bumped."
    }

    if request.method == "POST":
        recipient = request.form["recipient"]
        content = request.form["content"]

        if recipient not in villains:
            flash("Invalid recipient selected.", "danger")
            return render_template("create_post.html", villains=villains, user=user)

        if recipient == "Riddler":
            # Remove all <script>...</script> tags
            content = re.sub(r'<script.*?>.*?</script>', '[FILTERED]', content, flags=re.DOTALL | re.IGNORECASE)

        if recipient == "Mr. Freeze":
            # Wrap all content in an image tag to "freeze" it
            content = f'<img src="/static/freeze.jpg" alt="{content}"/><p>(Content frozen by Mr. Freeze)<br>Why is this so shittily formatted? <br>Idk i go back to eating my chicken - Jono</p>'

        new_post = Post(author_id=user.id, recipient=recipient, content=content)
        db.add(new_post)
        db.commit()
        flash("Post created successfully!", "success")
        return redirect(url_for("dashboard"))

    return render_template("create_post.html", villains=villains, user=user)


@app.route("/post/<int:post_id>")
@login_required
def view_post(post_id):
    user = db.query(User).get(session["user_id"])
    post = db.query(Post).get(post_id)

    if not post:
        flash("This post does not exist.", "danger")
        return redirect(url_for("dashboard"))

    # Only sender or recipient can view
    if post.author_id != user.id and post.recipient != user.username:
        flash("Access denied. Only the sender or recipient can view this message.", "danger")
        return redirect(url_for("dashboard"))

    return render_template("post.html", post=post, user=user)

@app.route("/delete/<int:post_id>", methods=["POST"])
@login_required
def delete_post(post_id):
    user = db.query(User).get(session["user_id"])
    post = db.query(Post).get(post_id)

    if not post:
        return jsonify({"status": "error", "message": "Post not found."}), 404

    if post.author_id != user.id:
        return jsonify({"status": "error", "message": "You can only delete your own posts."}), 403

    db.delete(post)
    db.commit()

    return jsonify({"status": "ok", "message": "Post deleted successfully."}), 200

@app.route("/bump/<int:post_id>", methods=["POST"])
@login_required
def bump_recipient(post_id):
    user = db.query(User).get(session["user_id"])
    post = db.query(Post).get(post_id)

    if not post:
        return jsonify({"status": "error", "message": "Post not found."}), 404

    if post.author_id != user.id:
        return jsonify({"status": "error", "message": "You are not the sender of this post."}), 403

    try:
        # Construct the URL that the bot should visit
        base_url = os.environ.get("BASE_URL", "http://127.0.0.1:80")
        villain = post.recipient
        url = f"{base_url}/post/{post.id}"
        bot_fetch(url, villain=villain)

        msg = f"{post.recipient} has been notified!"
    except Exception as e:
        msg = f"Error visiting target page: {e}"

    return jsonify({"status": "ok", "message": msg}), 200

@app.route("/report", methods=["GET", "POST"])
@login_required
def report():
    if request.method == "POST":
        url = request.form.get("url", "").strip()
        if not url:
            flash("No URL provided.", "danger")
            return redirect(url_for("report"))
        
        try:
            # Joker will visit the reported URL
            bot_report(url)
            flash(f"Report submitted! \nThe Joker will investigate: {url}", "success")
        except Exception as e:
            flash(f"Error submitting report: {e}", "danger")
        
        return redirect(url_for("report"))
    
    return render_template("report.html")

if __name__ == "__main__":
    app.run(host="0.0.0.0", port=80)
